"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.IgnoredQueriesLogic = void 0;
var _kea = require("kea");
var _i18n = require("@kbn/i18n");
var _constants = require("../../../../../../../shared/constants");
var _flash_messages = require("../../../../../../../shared/flash_messages");
var _http = require("../../../../../../../shared/http");
var _table_pagination = require("../../../../../../../shared/table_pagination");
var _engine = require("../../../../../engine");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const ALLOW_SUCCESS_MESSAGE = _i18n.i18n.translate('xpack.enterpriseSearch.appSearch.curations.ignoredSuggestionsPanel.allowQuerySuccessMessage', {
  defaultMessage: 'You’ll be notified about future suggestions for this query'
});
const IgnoredQueriesLogic = exports.IgnoredQueriesLogic = (0, _kea.kea)({
  path: ['enterprise_search', 'app_search', 'curations', 'ignored_queries_panel_logic'],
  actions: () => ({
    allowIgnoredQuery: ignoredQuery => ({
      ignoredQuery
    }),
    loadIgnoredQueries: true,
    onIgnoredQueriesLoad: (ignoredQueries, meta) => ({
      ignoredQueries,
      meta
    }),
    onPaginate: newPageIndex => ({
      newPageIndex
    })
  }),
  reducers: () => ({
    dataLoading: [true, {
      loadIgnoredQueries: () => true,
      onIgnoredQueriesLoad: () => false
    }],
    ignoredQueries: [[], {
      onIgnoredQueriesLoad: (_, {
        ignoredQueries
      }) => ignoredQueries
    }],
    meta: [{
      ..._constants.DEFAULT_META,
      page: {
        ..._constants.DEFAULT_META.page,
        size: 10
      }
    }, {
      onIgnoredQueriesLoad: (_, {
        meta
      }) => meta,
      onPaginate: (state, {
        newPageIndex
      }) => (0, _table_pagination.updateMetaPageIndex)(state, newPageIndex)
    }]
  }),
  listeners: ({
    actions,
    values
  }) => ({
    loadIgnoredQueries: async () => {
      const {
        meta
      } = values;
      const {
        http
      } = _http.HttpLogic.values;
      const {
        engineName
      } = _engine.EngineLogic.values;
      try {
        const response = await http.post(`/internal/app_search/engines/${engineName}/adaptive_relevance/suggestions`, {
          body: JSON.stringify({
            page: {
              current: meta.page.current,
              size: meta.page.size
            },
            filters: {
              status: ['disabled'],
              type: 'curation'
            }
          })
        });
        const queries = response.results.map(suggestion => suggestion.query);
        actions.onIgnoredQueriesLoad(queries, response.meta);
      } catch (e) {
        (0, _flash_messages.flashAPIErrors)(e);
      }
    },
    allowIgnoredQuery: async ({
      ignoredQuery
    }) => {
      const {
        http
      } = _http.HttpLogic.values;
      const {
        engineName
      } = _engine.EngineLogic.values;
      try {
        const response = await http.put(`/internal/app_search/engines/${engineName}/adaptive_relevance/suggestions`, {
          body: JSON.stringify([{
            query: ignoredQuery,
            type: 'curation',
            status: 'rejected'
          }])
        });
        if (Object.hasOwn(response.results[0], 'error')) {
          throw response.results[0].error;
        }
        (0, _flash_messages.flashSuccessToast)(ALLOW_SUCCESS_MESSAGE);
        // re-loading to update the current page rather than manually remove the query
        actions.loadIgnoredQueries();
      } catch (e) {
        (0, _flash_messages.flashAPIErrors)(e);
      }
    }
  })
});