"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.convertValidationFunction = exports.Type = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _joi = require("joi");
var _oas_meta_fields = require("../oas_meta_fields");
var _errors = require("../errors");
var _references = require("../references");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Meta fields used when introspecting runtime validation. Most notably for
 * generating OpenAPI spec.
 */

/**
 * Global validation Options to be provided when calling the `schema.validate()` method.
 */

/**
 * Options for dealing with unknown keys:
 * - allow: unknown keys will be permitted
 * - ignore: unknown keys will not fail validation, but will be stripped out
 * - forbid (default): unknown keys will fail validation
 */

const convertValidationFunction = validate => {
  return (value, {
    error
  }) => {
    let validationResultMessage;
    try {
      validationResultMessage = validate(value);
    } catch (e) {
      validationResultMessage = e.message || e;
    }
    if (typeof validationResultMessage === 'string') {
      return error('any.custom', {
        message: validationResultMessage
      });
    }
    return value;
  };
};
exports.convertValidationFunction = convertValidationFunction;
class Type {
  constructor(schema, options = {}) {
    // This is just to enable the `TypeOf` helper, and because TypeScript would
    // fail if it wasn't initialized we use a "trick" to which basically just
    // sets the value to `null` while still keeping the type.
    (0, _defineProperty2.default)(this, "type", null);
    // used for the `isConfigSchema` typeguard
    (0, _defineProperty2.default)(this, "__isKbnConfigSchemaType", true);
    /**
     * Internal "schema" backed by Joi.
     * @type {Schema}
     */
    (0, _defineProperty2.default)(this, "internalSchema", void 0);
    if (options.defaultValue !== undefined) {
      schema = schema.optional();

      // If default value is a function, then we must provide description for it.
      if (typeof options.defaultValue === 'function') {
        schema = schema.default(options.defaultValue);
      } else {
        schema = schema.default(_references.Reference.isReference(options.defaultValue) ? options.defaultValue.getSchema() : options.defaultValue);
      }
    }
    if (options.validate) {
      schema = schema.custom(convertValidationFunction(options.validate));
    }
    if (options.meta) {
      if (options.meta.description) {
        schema = schema.description(options.meta.description);
      }
      if (options.meta.deprecated) {
        schema = schema.meta({
          [_oas_meta_fields.META_FIELD_X_OAS_DEPRECATED]: true
        });
      }
      if (options.meta.deprecated && options.meta['x-discontinued']) schema = schema.meta({
        [_oas_meta_fields.META_FIELD_X_OAS_DISCONTINUED]: options.meta['x-discontinued']
      });
    }

    // Attach generic error handler only if it hasn't been attached yet since
    // only the last error handler is counted.
    if (schema.$_getFlag('error') === undefined) {
      schema = schema.error(([error]) => this.onError(error));
    }
    this.internalSchema = schema;
  }
  extendsDeep(newOptions) {
    return this;
  }

  /**
   * Validates the provided value against this schema.
   * If valid, the resulting output will be returned, otherwise an exception will be thrown.
   */
  validate(value, context = {}, namespace, validationOptions) {
    const {
      value: validatedValue,
      error
    } = this.internalSchema.validate(value, {
      context,
      presence: 'required',
      stripUnknown: {
        objects: (validationOptions === null || validationOptions === void 0 ? void 0 : validationOptions.stripUnknownKeys) === true
      }
    });
    if (error) {
      throw new _errors.ValidationError(error, namespace);
    }
    return validatedValue;
  }

  /**
   * @note intended for internal use, if you need to use this please contact
   *       the core team to discuss your use case.
   */
  getSchema() {
    return this.internalSchema;
  }
  getSchemaStructure() {
    return recursiveGetSchemaStructure(this.internalSchema);
  }
  handleError(type, context, path) {
    return undefined;
  }
  onError(error) {
    if (error instanceof _errors.SchemaTypeError) {
      return error;
    }
    const {
      local,
      code,
      path,
      value
    } = error;
    const convertedPath = path.map(entry => entry.toString());
    const context = {
      ...local,
      value
    };
    const errorHandleResult = this.handleError(code, context, convertedPath);
    if (errorHandleResult instanceof _errors.SchemaTypeError) {
      return errorHandleResult;
    }

    // If error handler just defines error message, then wrap it into proper
    // `SchemaTypeError` instance.
    if (typeof errorHandleResult === 'string') {
      return new _errors.SchemaTypeError(errorHandleResult, convertedPath);
    }

    // If error is produced by the custom validator, just extract source message
    // from context and wrap it into `SchemaTypeError` instance.
    if (code === 'any.custom' && context.message) {
      return new _errors.SchemaTypeError(context.message, convertedPath);
    }

    // `message` is only initialized once `toString` has been called (...)
    // see https://github.com/sideway/joi/blob/master/lib/errors.js
    const message = error.toString();
    return new _errors.SchemaTypeError(message || code, convertedPath);
  }
}
exports.Type = Type;
function recursiveGetSchemaStructure(internalSchema, path = []) {
  const array = [];
  // Note: we are relying on Joi internals to obtain the schema structure (recursive keys).
  // This is not ideal, but it works for now and we only need it for some integration test assertions.
  // If it breaks in the future, we'll need to update our tests.
  for (const [key, val] of internalSchema._ids._byKey.entries()) {
    array.push(...recursiveGetSchemaStructure(val.schema, [...path, key]));
  }
  if (!array.length) {
    let type;
    try {
      type = prettyPrintType(internalSchema, path);
    } catch (error) {
      // failed to find special type, might need to update for new joi versions or type usages
      type = internalSchema.type || 'unknown';
    }
    array.push({
      path,
      type
    });
  }
  return array;
}

/**
 * Returns a more accurate type from complex schema definitions.
 *
 * For example, conditional values resolve to type `any` when the nested value is only ever a `string`.
 *
 * @param internalSchema
 * @param path of current schema
 * @returns schema type
 */
function prettyPrintType(schema, path = []) {
  // takes array of possible values and de-dups and joins
  return [...new Set([prettyPrintTypeParts(schema, false, path)].flat())].filter(Boolean).join('|');
}

/**
 * Recursively collects all possible nested schema types.
 */
function prettyPrintTypeParts(schema, optional = false, path = []) {
  var _schema$_flags, _schema$_flags2, _schema$$_terms, _schema$$_terms$whens, _schema$$_terms2, _schema$$_terms2$matc, _schema$_flags3, _valids, _valids$_values;
  if (!(0, _joi.isSchema)(schema)) {
    if (schema === null) return 'null';
    return `${schema !== null && schema !== void 0 ? schema : 'unknown'}${optional ? '?' : ''}`;
  }
  const isOptionalType = optional || ((_schema$_flags = schema._flags) === null || _schema$_flags === void 0 ? void 0 : _schema$_flags.presence) === 'optional';
  // For explicit custom schema.never
  if (((_schema$_flags2 = schema._flags) === null || _schema$_flags2 === void 0 ? void 0 : _schema$_flags2.presence) === 'forbidden') return 'never';
  // For offeringBasedSchema, schema.when, schema.conditional
  if (((_schema$$_terms = schema.$_terms) === null || _schema$$_terms === void 0 ? void 0 : (_schema$$_terms$whens = _schema$$_terms.whens) === null || _schema$$_terms$whens === void 0 ? void 0 : _schema$$_terms$whens.length) > 0) return schema.$_terms.whens.flatMap(when => [when === null || when === void 0 ? void 0 : when.then, when === null || when === void 0 ? void 0 : when.otherwise].flatMap(s => prettyPrintTypeParts(s, isOptionalType, path)));
  // schema.oneOf, schema.allOf, etc.
  if (((_schema$$_terms2 = schema.$_terms) === null || _schema$$_terms2 === void 0 ? void 0 : (_schema$$_terms2$matc = _schema$$_terms2.matches) === null || _schema$$_terms2$matc === void 0 ? void 0 : _schema$$_terms2$matc.length) > 0) return schema.$_terms.matches.flatMap(s => prettyPrintTypeParts(s.schema, isOptionalType, path));
  // schema.literal
  if ((_schema$_flags3 = schema._flags) !== null && _schema$_flags3 !== void 0 && _schema$_flags3.only && ((_valids = schema._valids) === null || _valids === void 0 ? void 0 : (_valids$_values = _valids._values) === null || _valids$_values === void 0 ? void 0 : _valids$_values.size) > 0) return [...schema._valids._values.keys()].flatMap(v => prettyPrintTypeParts(v, isOptionalType, path));
  return `${(schema === null || schema === void 0 ? void 0 : schema.type) || 'unknown'}${isOptionalType ? '?' : ''}`;
}