"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerCspmUsageCollector = registerCspmUsageCollector;
var _indices_stats_collector = require("./indices_stats_collector");
var _resources_stats_collector = require("./resources_stats_collector");
var _schema = require("./schema");
var _accounts_stats_collector = require("./accounts_stats_collector");
var _rules_stats_collector = require("./rules_stats_collector");
var _installation_stats_collector = require("./installation_stats_collector");
var _alert_stats_collector = require("./alert_stats_collector");
var _cloud_accounts_stats_collector = require("./cloud_accounts_stats_collector");
var _muted_rules_stats_collector = require("./muted_rules_stats_collector");
var _constants = require("../../../../common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function registerCspmUsageCollector(logger, coreServices, usageCollection) {
  // usageCollection is an optional dependency, so make sure to return if it is not registered
  if (!usageCollection) {
    return;
  }

  // Create usage collector
  const cspmUsageCollector = usageCollection.makeUsageCollector({
    type: 'cloud_security_posture',
    isReady: async () => {
      await coreServices;
      return true;
    },
    fetch: async collectorFetchContext => {
      const awaitPromiseSafe = async (taskName, promise) => {
        try {
          const val = await promise;
          logger.info(`Cloud Security telemetry: ${taskName} payload was sent successfully`);
          return val;
        } catch (error) {
          logger.error(`${taskName} task failed: ${error.message}`);
          logger.error(error.stack);
          return error;
        }
      };
      const esClient = collectorFetchContext.esClient;
      const soClient = collectorFetchContext.soClient;
      const encryptedSoClient = (await coreServices)[0].savedObjects.createInternalRepository([_constants.INTERNAL_CSP_SETTINGS_SAVED_OBJECT_TYPE]);
      const [indicesStats, accountsStats, resourcesStats, rulesStats, installationStats, alertsStats, cloudAccountStats, mutedRulesStats] = await Promise.all([awaitPromiseSafe('Indices', (0, _indices_stats_collector.getIndicesStats)(esClient, soClient, coreServices, logger)), awaitPromiseSafe('Accounts', (0, _accounts_stats_collector.getAccountsStats)(esClient, logger)), awaitPromiseSafe('Resources', (0, _resources_stats_collector.getResourcesStats)(esClient, logger)), awaitPromiseSafe('Rules', (0, _rules_stats_collector.getRulesStats)(esClient, logger)), awaitPromiseSafe('Installation', (0, _installation_stats_collector.getInstallationStats)(esClient, soClient, coreServices, logger)), awaitPromiseSafe('Alerts', (0, _alert_stats_collector.getAlertsStats)(esClient, logger)), awaitPromiseSafe('Cloud Accounts', (0, _cloud_accounts_stats_collector.getAllCloudAccountsStats)(esClient, encryptedSoClient, logger)), awaitPromiseSafe('Muted Rules', (0, _muted_rules_stats_collector.getMutedRulesStats)(soClient, encryptedSoClient, logger))]);
      return {
        indices: indicesStats,
        accounts_stats: accountsStats,
        resources_stats: resourcesStats,
        rules_stats: rulesStats,
        installation_stats: installationStats,
        alerts_stats: alertsStats,
        cloud_account_stats: cloudAccountStats,
        muted_rules_stats: mutedRulesStats
      };
    },
    schema: _schema.cspmUsageSchema
  });

  // Register usage collector
  usageCollection.registerCollector(cspmUsageCollector);
}