"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.transformToUpdateScheme = exports.transformToCreateScheme = exports.transformESToAnonymizationFields = exports.transformESSearchToAnonymizationFields = exports.getUpdateScript = void 0;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const transformESToAnonymizationFields = response => {
  return response.map(anonymizationFieldSchema => {
    const anonymizationField = {
      timestamp: anonymizationFieldSchema['@timestamp'],
      createdAt: anonymizationFieldSchema.created_at,
      field: anonymizationFieldSchema.field,
      allowed: anonymizationFieldSchema.allowed,
      anonymized: anonymizationFieldSchema.anonymized,
      updatedAt: anonymizationFieldSchema.updated_at,
      namespace: anonymizationFieldSchema.namespace,
      id: anonymizationFieldSchema.id
    };
    return anonymizationField;
  });
};
exports.transformESToAnonymizationFields = transformESToAnonymizationFields;
const transformESSearchToAnonymizationFields = response => {
  return response.hits.hits.filter(hit => hit._source !== undefined).map(hit => {
    // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
    const anonymizationFieldSchema = hit._source;
    const anonymizationField = {
      timestamp: anonymizationFieldSchema['@timestamp'],
      createdAt: anonymizationFieldSchema.created_at,
      field: anonymizationFieldSchema.field,
      allowed: anonymizationFieldSchema.allowed,
      anonymized: anonymizationFieldSchema.anonymized,
      updatedAt: anonymizationFieldSchema.updated_at,
      namespace: anonymizationFieldSchema.namespace,
      // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
      id: hit._id
    };
    return anonymizationField;
  });
};
exports.transformESSearchToAnonymizationFields = transformESSearchToAnonymizationFields;
const transformToUpdateScheme = (user, updatedAt, {
  allowed,
  anonymized,
  id
}) => {
  return {
    id,
    updated_at: updatedAt,
    updated_by: user.username,
    allowed,
    anonymized
  };
};
exports.transformToUpdateScheme = transformToUpdateScheme;
const transformToCreateScheme = (user, createdAt, {
  allowed,
  anonymized,
  field
}) => {
  return {
    '@timestamp': createdAt,
    updated_at: createdAt,
    field,
    created_at: createdAt,
    created_by: user.username,
    allowed,
    anonymized
  };
};
exports.transformToCreateScheme = transformToCreateScheme;
const getUpdateScript = ({
  anonymizationField,
  isPatch
}) => {
  return {
    script: {
      source: `
    if (params.assignEmpty == true || params.containsKey('allowed')) {
      ctx._source.allowed = params.allowed;
    }
    if (params.assignEmpty == true || params.containsKey('anonymized')) {
      ctx._source.anonymized = params.anonymized;
    }
    ctx._source.updated_at = params.updated_at;
  `,
      lang: 'painless',
      params: {
        ...anonymizationField,
        // when assigning undefined in painless, it will remove property and wil set it to null
        // for patch we don't want to remove unspecified value in payload
        assignEmpty: !(isPatch !== null && isPatch !== void 0 ? isPatch : true)
      }
    }
  };
};
exports.getUpdateScript = getUpdateScript;