"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.resolveOverlapsAndMask = resolveOverlapsAndMask;
var _get_entity_mask = require("./get_entity_mask");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Processes detected matches by resolving overlaps and applying masks to records.
 * Returns updated state with masked records and anonymizations.
 */
function resolveOverlapsAndMask({
  detectedMatches,
  state,
  rules
}) {
  if (detectedMatches.length === 0) {
    return state;
  }

  // Group matches by record and field
  const matchesByRecordAndField = detectedMatches.reduce((acc, match) => {
    if (!acc.has(match.recordIndex)) {
      acc.set(match.recordIndex, new Map());
    }
    const recordMatches = acc.get(match.recordIndex);
    if (!recordMatches.has(match.recordKey)) {
      recordMatches.set(match.recordKey, []);
    }
    recordMatches.get(match.recordKey).push(match);
    return acc;
  }, new Map());

  // Process each record and field
  matchesByRecordAndField.forEach((fieldMatches, recordIndex) => {
    fieldMatches.forEach((matches, fieldName) => {
      const originalText = state.records[recordIndex][fieldName];

      // Resolve overlaps (first rule wins)
      const nonOverlappingMatches = [];
      let lastEnd = -1;
      for (const match of matches) {
        // Skip if this match overlaps with a higher priority match
        if (match.start < lastEnd) {
          continue;
        }
        nonOverlappingMatches.push(match);
        lastEnd = match.end;
      }

      // Apply masks in a single pass
      if (nonOverlappingMatches.length > 0) {
        let result = '';
        let lastIndex = 0;
        for (const match of nonOverlappingMatches) {
          // Add text before the match
          result += originalText.substring(lastIndex, match.start);

          // Create and add the mask
          const mask = (0, _get_entity_mask.getEntityMask)({
            value: match.matchValue,
            class_name: match.class_name
          });
          result += mask;

          // Add to anonymizations
          state.anonymizations.push({
            rule: {
              type: rules[match.ruleIndex].type
            },
            entity: {
              value: match.matchValue,
              class_name: match.class_name,
              mask
            }
          });
          lastIndex = match.end;
        }

        // Add remaining text after the last match
        result += originalText.substring(lastIndex);

        // Update the record with masked text
        state.records[recordIndex][fieldName] = result;
      }
    });
  });
  return state;
}