"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.fetchFieldCandidates = exports.TEXT_FIELD_SAFE_LIST = exports.SUPPORTED_ES_FIELD_TYPES_TEXT = exports.SUPPORTED_ES_FIELD_TYPES = void 0;
var _fieldTypes = require("@kbn/field-types");
var _ecs_fields = require("../ecs_fields");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// Supported field names for text fields for log rate analysis.
// If we analyse all detected text fields, we might run into performance
// issues with the `categorize_text` aggregation. Until this is resolved, we
// rely on a predefined white list of supported text fields.
const TEXT_FIELD_SAFE_LIST = exports.TEXT_FIELD_SAFE_LIST = ['message', 'error.message'];
const SUPPORTED_ES_FIELD_TYPES = exports.SUPPORTED_ES_FIELD_TYPES = [_fieldTypes.ES_FIELD_TYPES.KEYWORD, _fieldTypes.ES_FIELD_TYPES.IP
// Disabled boolean support because it causes problems with the `frequent_item_sets` aggregation
// ES_FIELD_TYPES.BOOLEAN,
];
const SUPPORTED_ES_FIELD_TYPES_TEXT = exports.SUPPORTED_ES_FIELD_TYPES_TEXT = [_fieldTypes.ES_FIELD_TYPES.TEXT, _fieldTypes.ES_FIELD_TYPES.MATCH_ONLY_TEXT];

// This override is meant to be used to force certain fields to be considered as
// text fields when both text and keyword type is available.

const fetchFieldCandidates = async ({
  esClient,
  abortSignal,
  arguments: args
}) => {
  const {
    textFieldCandidatesOverrides = [],
    ...params
  } = args;

  // Get all supported fields
  const respMapping = await esClient.fieldCaps({
    fields: '*',
    filters: '-metadata,-parent',
    include_empty_fields: false,
    index: params.index,
    index_filter: {
      range: {
        [params.timeFieldName]: {
          gte: params.deviationMin,
          lte: params.deviationMax
        }
      }
    },
    types: [...SUPPORTED_ES_FIELD_TYPES, ...SUPPORTED_ES_FIELD_TYPES_TEXT]
  }, {
    signal: abortSignal,
    maxRetries: 0
  });
  const allFieldNames = [];
  const acceptableFields = new Set();
  const acceptableTextFields = new Set();
  Object.entries(respMapping.fields).forEach(([key, value]) => {
    const fieldTypes = Object.keys(value);
    const isSupportedType = fieldTypes.some(type => SUPPORTED_ES_FIELD_TYPES.includes(type));
    const isAggregatable = fieldTypes.some(type => value[type].aggregatable);
    const isTextField = fieldTypes.some(type => SUPPORTED_ES_FIELD_TYPES_TEXT.includes(type));

    // Check if fieldName is something we can aggregate on
    if (isSupportedType && isAggregatable) {
      acceptableFields.add(key);
    }
    if (isTextField && TEXT_FIELD_SAFE_LIST.includes(key)) {
      acceptableTextFields.add(key);
    }
    allFieldNames.push(key);
  });
  const textFieldCandidatesOverridesWithKeywordPostfix = textFieldCandidatesOverrides.map(d => `${d}.keyword`);
  const keywordFieldCandidates = [...acceptableFields].filter(field => !textFieldCandidatesOverridesWithKeywordPostfix.includes(field));
  const textFieldCandidates = [...acceptableTextFields].filter(field => {
    const fieldName = field.replace(new RegExp(/\.text$/), '');
    return !keywordFieldCandidates.includes(fieldName) && !keywordFieldCandidates.includes(`${fieldName}.keyword`) || textFieldCandidatesOverrides.includes(field);
  });
  const isECS = (0, _ecs_fields.containsECSIdentifierField)(keywordFieldCandidates);
  return {
    isECS,
    // all keyword field candidates
    keywordFieldCandidates: keywordFieldCandidates.sort(),
    // preselection:
    // - if we identify an ECS schema, filter by custom ECS safe list
    // - if not, take the first 100 fields
    selectedKeywordFieldCandidates: isECS ? (0, _ecs_fields.filterByECSFields)(keywordFieldCandidates).sort() : keywordFieldCandidates.sort().slice(0, 100),
    // text field candidates
    textFieldCandidates: textFieldCandidates.sort(),
    selectedTextFieldCandidates: textFieldCandidates.sort()
  };
};
exports.fetchFieldCandidates = fetchFieldCandidates;