/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.cluster.routing.allocation;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.LongSupplier;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.support.RefCountingRunnable;
import org.elasticsearch.client.internal.Client;
import org.elasticsearch.cluster.ClusterInfo;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.DiskUsage;
import org.elasticsearch.cluster.block.ClusterBlockLevel;
import org.elasticsearch.cluster.metadata.IndexMetadata;
import org.elasticsearch.cluster.metadata.SingleNodeShutdownMetadata;
import org.elasticsearch.cluster.node.DiscoveryNode;
import org.elasticsearch.cluster.routing.RerouteService;
import org.elasticsearch.cluster.routing.RoutingNode;
import org.elasticsearch.cluster.routing.RoutingNodes;
import org.elasticsearch.cluster.routing.ShardRouting;
import org.elasticsearch.cluster.routing.allocation.DiskThresholdSettings;
import org.elasticsearch.cluster.routing.allocation.decider.DiskThresholdDecider;
import org.elasticsearch.common.Priority;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.settings.ClusterSettings;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.unit.ByteSizeValue;
import org.elasticsearch.common.util.concurrent.ConcurrentCollections;
import org.elasticsearch.core.Releasable;
import org.elasticsearch.gateway.GatewayService;
import org.elasticsearch.snapshots.SnapshotShardSizeInfo;

public class DiskThresholdMonitor {
    private static final Logger logger = LogManager.getLogger(DiskThresholdMonitor.class);
    private static final Settings READ_ONLY_ALLOW_DELETE_SETTINGS = Settings.builder().put(IndexMetadata.SETTING_READ_ONLY_ALLOW_DELETE, Boolean.TRUE.toString()).build();
    private static final Settings NOT_READ_ONLY_ALLOW_DELETE_SETTINGS = Settings.builder().putNull(IndexMetadata.SETTING_READ_ONLY_ALLOW_DELETE).build();
    private final DiskThresholdSettings diskThresholdSettings;
    private final Client client;
    private final Supplier<ClusterState> clusterStateSupplier;
    private final LongSupplier currentTimeMillisSupplier;
    private final RerouteService rerouteService;
    private final AtomicLong lastRunTimeMillis = new AtomicLong(Long.MIN_VALUE);
    private final AtomicBoolean checkInProgress = new AtomicBoolean();
    private final AtomicBoolean cleanupUponDisableCalled = new AtomicBoolean();
    private final Set<String> nodesOverLowThreshold = ConcurrentCollections.newConcurrentSet();
    private final Set<String> nodesOverHighThreshold = ConcurrentCollections.newConcurrentSet();
    private final Set<String> nodesOverHighThresholdAndRelocating = ConcurrentCollections.newConcurrentSet();
    private Set<String> lastNodes = Collections.emptySet();

    public DiskThresholdMonitor(Settings settings, Supplier<ClusterState> clusterStateSupplier, ClusterSettings clusterSettings, Client client, LongSupplier currentTimeMillisSupplier, RerouteService rerouteService) {
        this.clusterStateSupplier = clusterStateSupplier;
        this.currentTimeMillisSupplier = currentTimeMillisSupplier;
        this.rerouteService = rerouteService;
        this.diskThresholdSettings = new DiskThresholdSettings(settings, clusterSettings);
        this.client = client;
    }

    private void checkFinished() {
        boolean checkFinished = this.checkInProgress.compareAndSet(true, false);
        assert (checkFinished);
        logger.trace("checkFinished");
    }

    public void onNewInfo(ClusterInfo info) {
        ClusterState state = this.clusterStateSupplier.get();
        if (state.blocks().hasGlobalBlock(GatewayService.STATE_NOT_RECOVERED_BLOCK)) {
            logger.debug("skipping monitor as the cluster state is not recovered yet");
            return;
        }
        if (!this.checkInProgress.compareAndSet(false, true)) {
            logger.info("skipping monitor as a check is already in progress");
            return;
        }
        if (!this.diskThresholdSettings.isEnabled()) {
            this.removeExistingIndexBlocks();
            return;
        }
        this.cleanupUponDisableCalled.set(false);
        Map<String, DiskUsage> usages = info.getNodeLeastAvailableDiskUsages();
        if (usages == null) {
            logger.trace("skipping monitor as no disk usage information is available");
            this.lastNodes = Collections.emptySet();
            this.checkFinished();
            return;
        }
        logger.trace("processing new cluster info");
        boolean reroute = false;
        String explanation = "";
        long currentTimeMillis = this.currentTimeMillisSupplier.getAsLong();
        HashSet<String> nodes = new HashSet<String>(usages.keySet());
        DiskThresholdMonitor.cleanUpRemovedNodes(nodes, this.nodesOverLowThreshold);
        DiskThresholdMonitor.cleanUpRemovedNodes(nodes, this.nodesOverHighThreshold);
        DiskThresholdMonitor.cleanUpRemovedNodes(nodes, this.nodesOverHighThresholdAndRelocating);
        if (!this.lastNodes.equals(nodes)) {
            if (!this.lastNodes.containsAll(nodes)) {
                logger.debug("rerouting because disk usage info received from new nodes");
                reroute = true;
            }
            this.lastNodes = Collections.unmodifiableSet(nodes);
        }
        HashSet<String> indicesToMarkReadOnly = new HashSet<String>();
        RoutingNodes routingNodes = state.getRoutingNodes();
        HashSet<String> indicesNotToAutoRelease = new HashSet<String>();
        DiskThresholdMonitor.markNodesMissingUsageIneligibleForRelease(routingNodes, usages, indicesNotToAutoRelease);
        ArrayList<DiskUsage> usagesOverHighThreshold = new ArrayList<DiskUsage>();
        for (Map.Entry<String, DiskUsage> entry : usages.entrySet()) {
            String indexName;
            String node = entry.getKey();
            DiskUsage usage = entry.getValue();
            RoutingNode routingNode = routingNodes.node(node);
            ByteSizeValue total = ByteSizeValue.ofBytes(usage.totalBytes());
            if (DiskThresholdMonitor.isDedicatedFrozenNode(routingNode)) {
                if (usage.freeBytes() >= this.diskThresholdSettings.getFreeBytesThresholdFrozenFloodStage(total).getBytes()) continue;
                logger.warn("flood stage disk watermark [{}] exceeded on {}", (Object)this.diskThresholdSettings.describeFrozenFloodStageThreshold(total, false), (Object)usage);
                continue;
            }
            if (usage.freeBytes() < this.diskThresholdSettings.getFreeBytesThresholdFloodStage(total).getBytes()) {
                this.nodesOverLowThreshold.add(node);
                this.nodesOverHighThreshold.add(node);
                this.nodesOverHighThresholdAndRelocating.remove(node);
                if (routingNode != null) {
                    for (ShardRouting routing : routingNode) {
                        indexName = routing.index().getName();
                        indicesToMarkReadOnly.add(indexName);
                        indicesNotToAutoRelease.add(indexName);
                    }
                }
                logger.warn("flood stage disk watermark [{}] exceeded on {}, all indices on this node will be marked read-only", (Object)this.diskThresholdSettings.describeFloodStageThreshold(total, false), (Object)usage);
                continue;
            }
            if (usage.freeBytes() < this.diskThresholdSettings.getFreeBytesThresholdHighStage(total).getBytes() && routingNode != null) {
                for (ShardRouting routing : routingNode) {
                    indexName = routing.index().getName();
                    indicesNotToAutoRelease.add(indexName);
                }
            }
            long reservedSpace = info.getReservedSpace(usage.nodeId(), usage.path()).total();
            DiskUsage usageWithReservedSpace = new DiskUsage(usage.nodeId(), usage.nodeName(), usage.path(), usage.totalBytes(), Math.max(0L, usage.freeBytes() - reservedSpace));
            if (usageWithReservedSpace.freeBytes() < this.diskThresholdSettings.getFreeBytesThresholdHighStage(total).getBytes()) {
                this.nodesOverLowThreshold.add(node);
                this.nodesOverHighThreshold.add(node);
                if (this.lastRunTimeMillis.get() <= currentTimeMillis - this.diskThresholdSettings.getRerouteInterval().millis()) {
                    reroute = true;
                    explanation = "high disk watermark exceeded on one or more nodes";
                    usagesOverHighThreshold.add(usage);
                    continue;
                }
                logger.debug("high disk watermark exceeded on {} but an automatic reroute has occurred in the last [{}], skipping reroute", (Object)node, (Object)this.diskThresholdSettings.getRerouteInterval());
                continue;
            }
            if (usageWithReservedSpace.freeBytes() < this.diskThresholdSettings.getFreeBytesThresholdLowStage(total).getBytes()) {
                this.nodesOverHighThresholdAndRelocating.remove(node);
                boolean wasUnderLowThreshold = this.nodesOverLowThreshold.add(node);
                boolean wasOverHighThreshold = this.nodesOverHighThreshold.remove(node);
                assert (!(wasUnderLowThreshold && wasOverHighThreshold));
                if (wasUnderLowThreshold) {
                    logger.info("low disk watermark [{}] exceeded on {}, replicas will not be assigned to this node", (Object)this.diskThresholdSettings.describeLowThreshold(total, false), (Object)usage);
                    continue;
                }
                if (!wasOverHighThreshold) continue;
                logger.info("high disk watermark [{}] no longer exceeded on {}, but low disk watermark [{}] is still exceeded", (Object)this.diskThresholdSettings.describeHighThreshold(total, false), (Object)usage, (Object)this.diskThresholdSettings.describeLowThreshold(total, false));
                continue;
            }
            this.nodesOverHighThresholdAndRelocating.remove(node);
            if (!this.nodesOverLowThreshold.contains(node)) continue;
            if (this.lastRunTimeMillis.get() <= currentTimeMillis - this.diskThresholdSettings.getRerouteInterval().millis()) {
                reroute = true;
                explanation = "one or more nodes has gone under the high or low watermark";
                this.nodesOverLowThreshold.remove(node);
                this.nodesOverHighThreshold.remove(node);
                logger.info("low disk watermark [{}] no longer exceeded on {}", (Object)this.diskThresholdSettings.describeLowThreshold(total, false), (Object)usage);
                continue;
            }
            logger.debug("{} has gone below a disk threshold, but an automatic reroute has occurred in the last [{}], skipping reroute", (Object)node, (Object)this.diskThresholdSettings.getRerouteInterval());
        }
        try (RefCountingRunnable asyncRefs = new RefCountingRunnable(this::checkFinished);){
            if (reroute) {
                logger.debug("rerouting shards: [{}]", (Object)explanation);
                this.rerouteService.reroute("disk threshold monitor", Priority.HIGH, ActionListener.releaseAfter(ActionListener.runAfter(ActionListener.wrap(ignored -> {
                    ClusterState reroutedClusterState = this.clusterStateSupplier.get();
                    for (DiskUsage diskUsage : usagesOverHighThreshold) {
                        DiskUsage usageIncludingRelocations;
                        long relocatingShardsSize;
                        RoutingNode routingNode = reroutedClusterState.getRoutingNodes().node(diskUsage.nodeId());
                        if (routingNode != null) {
                            relocatingShardsSize = this.sizeOfRelocatingShards(routingNode, diskUsage, info, reroutedClusterState);
                            usageIncludingRelocations = new DiskUsage(diskUsage.nodeId(), diskUsage.nodeName(), diskUsage.path(), diskUsage.totalBytes(), diskUsage.freeBytes() - relocatingShardsSize);
                        } else {
                            usageIncludingRelocations = diskUsage;
                            relocatingShardsSize = 0L;
                        }
                        ByteSizeValue total = ByteSizeValue.ofBytes(usageIncludingRelocations.totalBytes());
                        if (usageIncludingRelocations.freeBytes() < this.diskThresholdSettings.getFreeBytesThresholdHighStage(total).getBytes()) {
                            this.nodesOverHighThresholdAndRelocating.remove(diskUsage.nodeId());
                            logger.warn("high disk watermark [{}] exceeded on {}, shards will be relocated away from this node; currently relocating away shards totalling [{}] bytes; the node is expected to continue to exceed the high disk watermark when these relocations are complete", (Object)this.diskThresholdSettings.describeHighThreshold(total, false), (Object)diskUsage, (Object)(-relocatingShardsSize));
                            continue;
                        }
                        if (this.nodesOverHighThresholdAndRelocating.add(diskUsage.nodeId())) {
                            logger.info("high disk watermark [{}] exceeded on {}, shards will be relocated away from this node; currently relocating away shards totalling [{}] bytes; the node is expected to be below the high disk watermark when these relocations are complete", (Object)this.diskThresholdSettings.describeHighThreshold(total, false), (Object)diskUsage, (Object)(-relocatingShardsSize));
                            continue;
                        }
                        logger.debug("high disk watermark [{}] exceeded on {}, shards will be relocated away from this node; currently relocating away shards totalling [{}] bytes", (Object)this.diskThresholdSettings.describeHighThreshold(total, false), (Object)diskUsage, (Object)(-relocatingShardsSize));
                    }
                }, e -> logger.debug("reroute failed", (Throwable)e)), this::setLastRunTimeMillis), asyncRefs.acquire()));
            } else {
                logger.trace("no reroute required");
            }
            Map nodeNameToIds = state.getRoutingNodes().stream().collect(Collectors.groupingBy(rn -> rn.node().getName(), Collectors.mapping(RoutingNode::nodeId, Collectors.toList())));
            Set routingNodeIds = state.getRoutingNodes().stream().map(RoutingNode::nodeId).collect(Collectors.toSet());
            Set nodesIdsPartOfReplacement = state.metadata().nodeShutdowns().getAll().values().stream().filter(meta -> meta.getType() == SingleNodeShutdownMetadata.Type.REPLACE).flatMap(meta -> Stream.concat(Stream.of(meta.getNodeId()), DiskThresholdMonitor.nodeIdsOrEmpty(meta, nodeNameToIds))).filter(routingNodeIds::contains).collect(Collectors.toSet());
            HashSet<String> indicesOnReplaceSourceOrTarget = new HashSet<String>();
            for (String nodeId : nodesIdsPartOfReplacement) {
                for (ShardRouting shardRouting : state.getRoutingNodes().node(nodeId)) {
                    indicesOnReplaceSourceOrTarget.add(shardRouting.index().getName());
                }
            }
            Set<String> indicesToAutoRelease = state.routingTable().indicesRouting().keySet().stream().filter(index -> !indicesNotToAutoRelease.contains(index)).filter(index -> state.getBlocks().hasIndexBlock((String)index, IndexMetadata.INDEX_READ_ONLY_ALLOW_DELETE_BLOCK)).filter(index -> !indicesOnReplaceSourceOrTarget.contains(index)).collect(Collectors.toSet());
            if (!indicesToAutoRelease.isEmpty()) {
                logger.info("releasing read-only block on indices " + String.valueOf(indicesToAutoRelease) + " since they are now allocated to nodes with sufficient disk space");
                this.updateIndicesReadOnly(indicesToAutoRelease, asyncRefs.acquire(), false);
            } else {
                logger.trace("no auto-release required");
            }
            indicesToMarkReadOnly.removeIf(index -> state.getBlocks().indexBlocked(ClusterBlockLevel.WRITE, (String)index));
            logger.trace("marking indices as read-only: [{}]", indicesToMarkReadOnly);
            if (!indicesToMarkReadOnly.isEmpty()) {
                this.updateIndicesReadOnly(indicesToMarkReadOnly, asyncRefs.acquire(), true);
            }
        }
    }

    private static Stream<String> nodeIdsOrEmpty(SingleNodeShutdownMetadata meta, Map<String, List<String>> nodeNameToIds) {
        List<String> ids = nodeNameToIds.get(meta.getTargetNodeName());
        return ids == null ? Stream.empty() : ids.stream();
    }

    long sizeOfRelocatingShards(RoutingNode routingNode, DiskUsage diskUsage, ClusterInfo info, ClusterState reroutedClusterState) {
        return DiskThresholdDecider.sizeOfUnaccountedShards(routingNode, true, diskUsage.path(), info, SnapshotShardSizeInfo.EMPTY, reroutedClusterState.metadata(), reroutedClusterState.routingTable(), 0L);
    }

    private static void markNodesMissingUsageIneligibleForRelease(RoutingNodes routingNodes, Map<String, DiskUsage> usages, Set<String> indicesToMarkIneligibleForAutoRelease) {
        for (RoutingNode routingNode : routingNodes) {
            if (usages.containsKey(routingNode.nodeId())) continue;
            for (ShardRouting routing : routingNode) {
                String indexName = routing.index().getName();
                indicesToMarkIneligibleForAutoRelease.add(indexName);
            }
        }
    }

    private void setLastRunTimeMillis() {
        this.lastRunTimeMillis.getAndUpdate(l -> Math.max(l, this.currentTimeMillisSupplier.getAsLong()));
    }

    protected void updateIndicesReadOnly(Set<String> indicesToUpdate, Releasable onCompletion, boolean readOnly) {
        Settings readOnlySettings = readOnly ? READ_ONLY_ALLOW_DELETE_SETTINGS : NOT_READ_ONLY_ALLOW_DELETE_SETTINGS;
        this.client.admin().indices().prepareUpdateSettings(indicesToUpdate.toArray(Strings.EMPTY_ARRAY)).setSettings(readOnlySettings).origin("disk-threshold-monitor").execute(ActionListener.releaseAfter(ActionListener.runAfter(ActionListener.noop().delegateResponse((l, e) -> logger.debug(() -> "setting indices [" + readOnly + "] read-only failed", (Throwable)e)), this::setLastRunTimeMillis), onCompletion));
    }

    private void removeExistingIndexBlocks() {
        if (this.cleanupUponDisableCalled.get()) {
            this.checkFinished();
            return;
        }
        ActionListener<Void> wrappedListener = ActionListener.wrap(r -> {
            this.cleanupUponDisableCalled.set(true);
            this.checkFinished();
        }, e -> {
            logger.debug("removing read-only blocks from indices failed", (Throwable)e);
            this.checkFinished();
        });
        ClusterState state = this.clusterStateSupplier.get();
        Set<String> indicesToRelease = state.getBlocks().indices().keySet().stream().filter(index -> state.getBlocks().hasIndexBlock((String)index, IndexMetadata.INDEX_READ_ONLY_ALLOW_DELETE_BLOCK)).collect(Collectors.toUnmodifiableSet());
        logger.trace("removing read-only block from indices [{}]", indicesToRelease);
        if (!indicesToRelease.isEmpty()) {
            this.client.admin().indices().prepareUpdateSettings(indicesToRelease.toArray(Strings.EMPTY_ARRAY)).setSettings(NOT_READ_ONLY_ALLOW_DELETE_SETTINGS).origin("disk-threshold-monitor").execute(wrappedListener.map(r -> null));
        } else {
            wrappedListener.onResponse(null);
        }
    }

    private static void cleanUpRemovedNodes(Set<String> nodesToKeep, Set<String> nodesToCleanUp) {
        for (String node : nodesToCleanUp) {
            if (nodesToKeep.contains(node)) continue;
            nodesToCleanUp.remove(node);
        }
    }

    private static boolean isDedicatedFrozenNode(RoutingNode routingNode) {
        if (routingNode == null) {
            return false;
        }
        DiscoveryNode node = routingNode.node();
        return node.isDedicatedFrozenNode();
    }
}

