# frozen_string_literal: true

module Actastic
  module Dirty
    extend ActiveSupport::Concern
    include ActiveModel::AttributeMethods

    # --------------------------------------------------------------------------
    # Rip out ActiveModel::Dirty from Rails 5.1.7 since it was working for us.
    # Rails 5.2.x ActiveModel::Dirty introduces significant breaking changes.
    # https://github.com/rails/rails/blob/v5.1.7/activemodel/lib/active_model/dirty.rb

    included do
      attribute_method_suffix '_changed?', '_change', '_will_change!', '_was'
      attribute_method_suffix '_previously_changed?', '_previous_change'
      attribute_method_affix prefix: 'restore_', suffix: '!'
    end

    # Returns a hash of changed attributes indicating their original
    # and new values like <tt>attr => [original value, new value]</tt>.
    #
    #   person.changes # => {}
    #   person.name = 'bob'
    #   person.changes # => { "name" => ["bill", "bob"] }
    def changes
      ActiveSupport::HashWithIndifferentAccess[changed.map { |attr| [attr, attribute_change(attr)] }]
    end

    # Returns a hash of attributes that were changed before the model was saved.
    #
    #   person.name # => "bob"
    #   person.name = 'robert'
    #   person.save
    #   person.previous_changes # => {"name" => ["bob", "robert"]}
    def previous_changes
      @previously_changed ||= ActiveSupport::HashWithIndifferentAccess.new
    end

    # Returns a hash of the attributes with unsaved changes indicating their original
    # values like <tt>attr => original value</tt>.
    #
    #   person.name # => "bob"
    #   person.name = 'robert'
    #   person.changed_attributes # => {"name" => "bob"}
    def changed_attributes
      @changed_attributes ||= ActiveSupport::HashWithIndifferentAccess.new
    end

    # Handles <tt>*_previously_changed?</tt> for +method_missing+.
    def attribute_previously_changed?(attr) # :nodoc:
      previous_changes_include?(attr)
    end

    # Restore all previous data of the provided attributes.
    def restore_attributes(attributes = changed)
      attributes.each { |attr| restore_attribute! attr }
    end

    # --------------------------------------------------------------------------

    def changed
      changed_attributes.keys.select { |attr| attribute_change(attr) }
    end

    def changed?
      changed_attributes.keys.any? { |attr| attribute_changed?(attr) }
    end

    # Prevents a SystemStackError when fetching a Hash or Array
    def attribute_changed?(attr, _options = {})
      return false unless changes_include?(attr)
      changed_attributes[attr] != @attributes[attr.to_sym]
    end

    def new_changes
      changed.each_with_object({}) do |attr, memo|
        field = attr.to_sym
        memo[field] = @attributes[field]
      end
    end

    def changes_json
      Actastic.configuration.json_encoder_class.dump(:doc => new_changes.transform_values do |value|
        (value.is_a?(Hash) ? [value] : value)
      end)
    end

    def attribute_was(attr)
      attribute_changed?(attr) ? changed_attributes[attr] : @attributes[attr.to_sym]
    end

    protected

    def attribute_method?(attr_name)
      respond_to_without_attributes?(:attributes) && self.class.field?(attr_name.to_sym)
    end

    private

    # --------------------------------------------------------------------------
    # Rip out ActiveModel::Dirty from Rails 5.1.7 since it was working for us.
    # Rails 5.2.x ActiveModel::Dirty introduces significant breaking changes
    # https://github.com/rails/rails/blob/v5.1.7/activemodel/lib/active_model/dirty.rb

    # Returns +true+ if attr_name is changed, +false+ otherwise.
    def changes_include?(attr_name)
      attributes_changed_by_setter.include?(attr_name)
    end
    alias :attribute_changed_by_setter? :changes_include?

    # Returns +true+ if attr_name were changed before the model was saved,
    # +false+ otherwise.
    def previous_changes_include?(attr_name)
      previous_changes.include?(attr_name)
    end

    # Removes current changes and makes them accessible through +previous_changes+.
    def changes_applied
      @previously_changed = changes
      @changed_attributes = ActiveSupport::HashWithIndifferentAccess.new
    end

    # Clears all dirty data: current changes and previous changes.
    def clear_changes_information
      @previously_changed = ActiveSupport::HashWithIndifferentAccess.new
      @changed_attributes = ActiveSupport::HashWithIndifferentAccess.new
    end

    # Handles <tt>*_previous_change</tt> for +method_missing+.
    def attribute_previous_change(attr)
      previous_changes[attr] if attribute_previously_changed?(attr)
    end

    # Handles <tt>restore_*!</tt> for +method_missing+.
    def restore_attribute!(attr)
      if attribute_changed?(attr)
        __send__("#{attr}=", changed_attributes[attr])
        clear_attribute_changes([attr])
      end
    end

    # This is necessary because `changed_attributes` might be overridden in
    # other implementations (e.g. in `ActiveRecord`)
    alias_method :attributes_changed_by_setter, :changed_attributes # :nodoc:

    # Force an attribute to have a particular "before" value
    def set_attribute_was(attr, old_value)
      attributes_changed_by_setter[attr] = old_value
    end

    # Remove changes information for the provided attributes.
    def clear_attribute_changes(attributes)
      attributes_changed_by_setter.except!(*attributes)
    end

    # --------------------------------------------------------------------------

    def attribute_change(attr)
      [changed_attributes[attr], @attributes[attr.to_sym]] if attribute_changed?(attr)
    end

    # Prevents a SystemStackError when there is an assignment to the same attribute in an accessor
    def attribute_will_change!(attr)
      return if attribute_changed?(attr)

      begin
        value = @attributes[attr.to_sym]
        value = value.duplicable? ? value.clone : value
      rescue TypeError, NoMethodError
        # ignore
      end

      set_attribute_was(attr, value)
    end
  end
end
