# frozen_string_literal: true
module Actastic
  module Serialization
    extend ActiveSupport::Concern

    class Encryptor
      class << self
        def dump(value)
          encrypt(value)
        end

        def load(value)
          decrypt(value)
        end

        delegate :encrypt, :decrypt, :to => :'SecretKeeper::Backend::Elasticsearch'
      end
    end

    class SearchableEncryptor
      class << self
        def dump(value)
          {
            :hash => Digest::SHA2.hexdigest(value),
            :ciphertext => Actastic::Serialization::Encryptor.dump(value)
          }
        end

        def load(value)
          Actastic::Serialization::Encryptor.load(value&.symbolize_keys&.fetch(:ciphertext))
        end
      end
    end

    class_methods do
      def serialize(attribute, serializer)
        _actastic_serializers[attribute] = serializer
      end

      def serialize_attribute(attribute, value)
        return value unless serializer = _actastic_serializers[attribute]
        serializer.dump(value)
      end

      def deserialize_attribute(attribute, value)
        return value unless serializer = _actastic_serializers[attribute]
        serializer.load(value)
      end

      private

      def _actastic_serializers
        @_actastic_serializers ||= HashWithIndifferentAccess.new
      end
    end
  end
end
