/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

package org.elasticsearch.xpack.core.ml.inference.trainedmodel;

import org.elasticsearch.TransportVersion;
import org.elasticsearch.TransportVersions;
import org.elasticsearch.common.io.stream.StreamInput;
import org.elasticsearch.common.io.stream.StreamOutput;
import org.elasticsearch.core.Nullable;
import org.elasticsearch.xcontent.ConstructingObjectParser;
import org.elasticsearch.xcontent.ParseField;
import org.elasticsearch.xcontent.XContentBuilder;
import org.elasticsearch.xcontent.XContentParser;
import org.elasticsearch.xpack.core.ml.MlConfigVersion;
import org.elasticsearch.xpack.core.ml.inference.persistence.InferenceIndexConstants;
import org.elasticsearch.xpack.core.ml.utils.ExceptionsHelper;
import org.elasticsearch.xpack.core.ml.utils.NamedXContentObjectHelper;

import java.io.IOException;
import java.util.Objects;
import java.util.Optional;

public class TextEmbeddingConfig implements NlpConfig {

    public static final String NAME = "text_embedding";

    public static ParseField EMBEDDING_SIZE = new ParseField("embedding_size");

    public static TextEmbeddingConfig fromXContentStrict(XContentParser parser) {
        return STRICT_PARSER.apply(parser, null);
    }

    public static TextEmbeddingConfig fromXContentLenient(XContentParser parser) {
        return LENIENT_PARSER.apply(parser, null);
    }

    private static final ConstructingObjectParser<TextEmbeddingConfig, Void> STRICT_PARSER = createParser(false);
    private static final ConstructingObjectParser<TextEmbeddingConfig, Void> LENIENT_PARSER = createParser(true);

    private static ConstructingObjectParser<TextEmbeddingConfig, Void> createParser(boolean ignoreUnknownFields) {
        ConstructingObjectParser<TextEmbeddingConfig, Void> parser = new ConstructingObjectParser<>(
            NAME,
            ignoreUnknownFields,
            a -> TextEmbeddingConfig.create((VocabularyConfig) a[0], (Tokenization) a[1], (String) a[2], (Integer) a[3])
        );
        parser.declareObject(ConstructingObjectParser.optionalConstructorArg(), (p, c) -> {
            if (ignoreUnknownFields == false) {
                throw ExceptionsHelper.badRequestException(
                    "illegal setting [{}] on inference model creation",
                    VOCABULARY.getPreferredName()
                );
            }
            return VocabularyConfig.fromXContentLenient(p);
        }, VOCABULARY);
        parser.declareNamedObject(
            ConstructingObjectParser.optionalConstructorArg(),
            (p, c, n) -> p.namedObject(Tokenization.class, n, ignoreUnknownFields),
            TOKENIZATION
        );
        parser.declareString(ConstructingObjectParser.optionalConstructorArg(), RESULTS_FIELD);
        parser.declareInt(ConstructingObjectParser.optionalConstructorArg(), EMBEDDING_SIZE);
        return parser;
    }

    private final VocabularyConfig vocabularyConfig;
    private final Tokenization tokenization;
    private final String resultsField;
    private final Integer embeddingSize;

    static TextEmbeddingConfig create(
        @Nullable VocabularyConfig vocabularyConfig,
        @Nullable Tokenization tokenization,
        @Nullable String resultsField,
        @Nullable Integer embeddingSize
    ) {
        var config = new TextEmbeddingConfig(
            Optional.ofNullable(vocabularyConfig).orElse(new VocabularyConfig(InferenceIndexConstants.nativeDefinitionStore())),
            tokenization == null ? Tokenization.createDefault() : tokenization,
            resultsField,
            embeddingSize
        );

        if (config.embeddingSize != null && config.embeddingSize <= 0) {
            throw ExceptionsHelper.badRequestException(
                "[{}] must be a number greater than 0; configured size [{}]",
                EMBEDDING_SIZE.getPreferredName(),
                embeddingSize
            );
        }
        if (config.tokenization.span != -1) {
            throw ExceptionsHelper.badRequestException(
                "[{}] does not support windowing long text sequences; configured span [{}]",
                NAME,
                config.tokenization.span
            );
        }

        return config;
    }

    private TextEmbeddingConfig(VocabularyConfig vocabularyConfig, Tokenization tokenization, String resultsField, Integer embeddingSize) {
        this.vocabularyConfig = vocabularyConfig;
        this.tokenization = tokenization;
        this.resultsField = resultsField;
        this.embeddingSize = embeddingSize;
    }

    public TextEmbeddingConfig(StreamInput in) throws IOException {
        vocabularyConfig = new VocabularyConfig(in);
        tokenization = in.readNamedWriteable(Tokenization.class);
        resultsField = in.readOptionalString();
        if (in.getTransportVersion().onOrAfter(TransportVersions.V_8_8_0)) {
            embeddingSize = in.readOptionalVInt();
        } else {
            embeddingSize = null;
        }
    }

    @Override
    public XContentBuilder toXContent(XContentBuilder builder, Params params) throws IOException {
        builder.startObject();
        builder.field(VOCABULARY.getPreferredName(), vocabularyConfig, params);
        NamedXContentObjectHelper.writeNamedObject(builder, params, TOKENIZATION.getPreferredName(), tokenization);
        if (resultsField != null) {
            builder.field(RESULTS_FIELD.getPreferredName(), resultsField);
        }
        if (embeddingSize != null) {
            builder.field(EMBEDDING_SIZE.getPreferredName(), embeddingSize);
        }
        builder.endObject();
        return builder;
    }

    @Override
    public String getWriteableName() {
        return NAME;
    }

    @Override
    public void writeTo(StreamOutput out) throws IOException {
        vocabularyConfig.writeTo(out);
        out.writeNamedWriteable(tokenization);
        out.writeOptionalString(resultsField);
        if (out.getTransportVersion().onOrAfter(TransportVersions.V_8_8_0)) {
            out.writeOptionalVInt(embeddingSize);
        }
    }

    @Override
    public boolean isTargetTypeSupported(TargetType targetType) {
        return false;
    }

    @Override
    public InferenceConfig apply(InferenceConfigUpdate update) {
        if (update instanceof TextEmbeddingConfigUpdate configUpdate) {
            return new TextEmbeddingConfig(
                vocabularyConfig,
                configUpdate.tokenizationUpdate == null ? tokenization : configUpdate.tokenizationUpdate.apply(tokenization),
                configUpdate.getResultsField() == null ? resultsField : configUpdate.getResultsField(),
                embeddingSize
            );
        } else if (update instanceof TokenizationConfigUpdate tokenizationUpdate) {
            var updatedTokenization = getTokenization().updateWindowSettings(tokenizationUpdate.getSpanSettings());
            return new TextEmbeddingConfig(vocabularyConfig, updatedTokenization, resultsField, embeddingSize);
        } else {
            throw incompatibleUpdateException(update.getName());
        }
    }

    @Override
    public MlConfigVersion getMinimalSupportedMlConfigVersion() {
        return MlConfigVersion.V_8_0_0;
    }

    @Override
    public TransportVersion getMinimalSupportedTransportVersion() {
        return TransportVersions.V_8_0_0;
    }

    @Override
    public boolean isAllocateOnly() {
        return true;
    }

    @Override
    public String getName() {
        return NAME;
    }

    @Override
    public boolean equals(Object o) {
        if (o == this) return true;
        if (o == null || getClass() != o.getClass()) return false;

        TextEmbeddingConfig that = (TextEmbeddingConfig) o;
        return Objects.equals(vocabularyConfig, that.vocabularyConfig)
            && Objects.equals(tokenization, that.tokenization)
            && Objects.equals(resultsField, that.resultsField)
            && Objects.equals(embeddingSize, that.embeddingSize);
    }

    @Override
    public int hashCode() {
        return Objects.hash(vocabularyConfig, tokenization, resultsField, embeddingSize);
    }

    @Override
    public VocabularyConfig getVocabularyConfig() {
        return vocabularyConfig;
    }

    @Override
    public Tokenization getTokenization() {
        return tokenization;
    }

    @Override
    public String getResultsField() {
        return resultsField;
    }

    public Integer getEmbeddingSize() {
        return embeddingSize;
    }
}
